/* RBinterface.cp */

#include "rb_plugin.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "myCalc.h"

// WinHeader.h

static long myPrecision;

typedef struct
{
	double					Re;
	double					Im;
	
} ComplexData;



static void complexMultiply(ComplexData& z, const ComplexData& x, const ComplexData& y)
{
	double			ac, bd;
	
	if(x.Re!=0 && x.Im!=0 && y.Re!=0 && y.Im!=0)
	{
		ac = x.Re*y.Re;
		bd = x.Im*y.Im;
		
		// the order here is important since z might be the same as x or y
		z.Im = (x.Re + x.Im)*(y.Re + y.Im) - ac - bd;
		z.Re = ac - bd;
		return;
	}
	
	if(x.Re==0)
	{
		ac = -x.Im*y.Im;
		z.Im = x.Im*y.Re;
		z.Re = ac;
		return;
	}
	
	if(x.Im==0)
	{
		ac = x.Re*y.Re;
		z.Im = x.Re*y.Im;
		z.Re = ac;
		return;
	}
	
	if(y.Re==0)
	{
		ac = -x.Im*y.Im;
		z.Im = x.Re*y.Im;
		z.Re = ac;
		return;
	}
	
	if(y.Im==0)
	{
		ac = x.Re*y.Re;
		z.Im = x.Im*y.Re;
		z.Re = ac;
		return;
	}
	
}/* complexMultiply */


static void complexDivide(ComplexData& z, const ComplexData& x, const ComplexData& y)
{
	double			ac, bd, den;
	
	// Re[(a + ib)/(c + id)] = (ac + bd)/(c*c + d*d)
	// Im[(a + ib)/(c + id)] = (bc - ad)/(c*c + d*d) = [(a + b)(c - d) - ac + bd]/(c*c + d*d)
	
	// a = x.Re, b = x.Im, c = y.Re, d = y.Im
	
	if(y.Re==0 && y.Im==0)
	{
		z.Re = NAN;
		z.Im = NAN;
		return;
	}
	
	if(x.Im==0 && y.Im==0)
	{
		z.Re = x.Re/y.Re;
		z.Im = 0;
		return;
	}
	
	if(x.Re==0 && y.Re==0)
	{
		z.Re = x.Im/y.Im;
		z.Im = 0;
		return;
	}
	
	if(x.Re==0 && y.Im==0)
	{
		z.Re = 0;
		z.Im = x.Im/y.Re;
		return;
	}
	
	if(x.Im==0 && y.Re==0)
	{
		z.Re = 0;
		z.Im = -x.Re/y.Im;
		return;
	}
	
	den = y.Re*y.Re + y.Im*y.Im;
	
	ac = x.Re*y.Re;
	bd = x.Im*y.Im;
	
	// the order here is important since z might be the same as x or y
	z.Im = ((x.Re + x.Im)*(y.Re - y.Im) - ac + bd)/den;
	z.Re = (ac + bd)/den;
	
}/* complexDivide */


extern REALclassDefinition		ComplexClass;

/* -------------------------------------
 ComplexConstructor
 
 Called by REALbasic when a new Complex object is instantiated.
 
 instance = the new Complex object
 ------------------------------------- */

static void ComplexConstructor(REALobject instance)
{
	ComplexData		*data;
	
	// Get the class data
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (data == nil)
		return;
	
	data->Re = 0;
	data->Im = 0;
	
}/* ComplexConstructor */


static void NewComplex(REALobject instance, double xRe, double xIm)
{
	ComplexData		*data;
	
	// Get the class data
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (data == nil)
		return;
	
	data->Re = xRe;
	data->Im = xIm;
	
}/* NewComplex */


static void NewComplexD(REALobject instance, double x)
{
	ComplexData		*data;
	
	// Get the class data
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (data == nil)
		return;
	
	data->Re = x;
	data->Im = 0;
	
}/* NewComplexD */


static void NewComplexS(REALobject instance, REALstring CStr)
{
	ComplexData		*data;
	long			k, r, length;
	char			*dataString, *doubleString, ch;
	bool			endOfData;
	double			theDouble;
	
	endOfData = false;
	
	// Get the class data
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (data == nil)
		return;
	
	// now get the complex number from CStr
	length = CStr->Length();
	
	// if length==0 make it a null complex number and return
	if(length==0)
	{
		data->Re = 0;
		data->Im = 0;
		return;
	}
	
	// length!=0
	dataString = (char *)malloc((length+1)*sizeof(char));
	doubleString = (char *)malloc((length+1)*sizeof(char));
	memcpy(dataString, (char*)CStr->CString(), length); // to, from
	dataString[length] = 0; // makes it a C string
	
	k = 0; // start index of dataString
	
	if(k>=length)
		endOfData = true;
	
	if(!endOfData)
	{
		r = 0; // start index of doubleString
		// get jth double string as doubleString, read to , or end of rowString
		ch = dataString[k];
		while(k<length && ch!=',')
		{
			doubleString[r] = ch;
			r++;
			k++;
			ch = dataString[k]; 
		}
		
		doubleString[r] = 0;
		
		if(k>=length)
			endOfData = true;
		else
			k++; // skip over ","
		
		if(!(sscanf(doubleString, "%lf", &theDouble)==1))
			theDouble = 0;
		
		data->Re = theDouble;
	}
	else
		data->Re = 0;
	
	if(!endOfData)
	{
		r = 0; // start index of doubleString
		// get jth double string as doubleString, read to , or end of rowString
		ch = dataString[k];
		while(k<length && ch!=',')
		{
			doubleString[r] = ch;
			r++;
			k++;
			ch = dataString[k]; 
		}
		
		doubleString[r] = 0;
		
		if(!(sscanf(doubleString, "%lf", &theDouble)==1))
			theDouble = 0;
		
		data->Im = theDouble;
	}
	else
		data->Im = theDouble;
	
	free(dataString);
	free(doubleString);
	
}/* NewComplexS */


/* -------------------------------------
 ComplexDestructor
 
 Called by REALbasic when a Complex object is destroyed.
 
 instance = the Complex object being destroyed
 ------------------------------------- */

static void ComplexDestructor(REALobject instance)
{
	
}/* ComplexDestructor */

/*
 static REALobject Complex(double xRe, double xIm)
 {
 REALobject		z;
 ComplexData		*zData;
 
 z = REALnewInstance("Complex");
 zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
 
 zData->Re = xRe;
 zData->Im = xIm;
 
 return z;
 
 }*//* Complex */


static REALobject ComplexNegate(REALobject instance)
{
	REALobject		z;
	ComplexData		*data, *zData;
	
	if(instance==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (data == nil)
		return instance;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = -data->Re;
	zData->Im = -data->Im;
	
	return z;
	
}/* ComplexNegate */


static double ComplexReal(REALobject instance)
{
	ComplexData		*data;
	
	if(instance==nil)
	{
		return NAN;
	}
	
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (data == nil)
		return NAN;
	
	return data->Re;
	
}/* ComplexReal */


static double ComplexImag(REALobject instance)
{
	ComplexData		*data;
	
	if(instance==nil)
	{
		return NAN;
	}
	
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (data == nil)
		return NAN;
	
	return data->Im;
	
}/* ComplexImag */


static REALstring ComplexStr(REALobject instance)
{
	ComplexData		*data;
	long			length;
	char			myString[100];
	char			formatString1[10], formatString[15], precString[5];
	
	if(instance==nil)
	{
		REALstring str=REALBuildString (nil, 0);
		return str;
	}
	
	strcpy(formatString1, "%.");
	sprintf(precString, "%d", myPrecision);
	strcat(formatString1, precString);
	strcat(formatString1, "g");
	
	strcpy(formatString, formatString1);
	strcat(formatString, ", ");
	strcat(formatString, formatString1);
	
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (data == nil)
		strcpy(myString, "");
	else
		sprintf(myString, formatString, data->Re, data->Im);
	
	length = strlen(myString);
	REALstring str=REALBuildString (nil, length);
	memcpy ((char*)str->CString(), myString, length); // to, from
	
	return str;
	
}/* ComplexStr */

static REALstring ComplexStr2(REALobject instance)
{
	ComplexData		*data;
	long			length;
	char			myString[100];
	char			formatString1[10], formatString[15], precString[5];
	
	if(instance==nil)
	{
		REALstring str=REALBuildString (nil, 0);
		return str;
	}
	
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	if (data == nil)
	{
		strcpy(myString, "");
		goto endGame2;
	}
	
	strcpy(formatString1, "%.");
	sprintf(precString, "%d", myPrecision);
	strcat(formatString1, precString);
	strcat(formatString1, "g");
	
	strcpy(formatString, formatString1);
	if(data->Im>=0)
		strcat(formatString, " + i ");
	else
		strcat(formatString, " - i ");
	strcat(formatString, formatString1);
	
	sprintf(myString, formatString, data->Re, abs(data->Im));
	
endGame2:
	
	length = strlen(myString);
	REALstring str=REALBuildString (nil, length);
	memcpy ((char*)str->CString(), myString, length); // to, from
	
	return str;
	
}/* ComplexStr2 */


static REALstring ComplexStr2EE(REALobject instance)
{
	ComplexData		*data;
	long			length;
	char			myString[100];
	char			formatString1[10], formatString[15], precString[5];
	
	if(instance==nil)
	{
		REALstring str=REALBuildString (nil, 0);
		return str;
	}
	
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	if (data == nil)
	{
		strcpy(myString, "");
		goto endGame2EE;
	}
	
	strcpy(formatString1, "%.");
	sprintf(precString, "%d", myPrecision);
	strcat(formatString1, precString);
	strcat(formatString1, "g");
	
	strcpy(formatString, formatString1);
	if(data->Im>=0)
		strcat(formatString, " + j ");
	else
		strcat(formatString, " - j ");
	strcat(formatString, formatString1);
	
	sprintf(myString, formatString, data->Re, abs(data->Im));
	
endGame2EE:
	
	length = strlen(myString);
	REALstring str=REALBuildString (nil, length);
	memcpy ((char*)str->CString(), myString, length); // to, from
	
	return str;
	
}/* ComplexStr2EE */


/*
 static void EquateC(REALobject instance, double xRe, double xIm)
 {
 ComplexData		*data;
 data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
 
 if (data == nil)
 return;
 
 data->Re = xRe;
 data->Im = xIm;
 
 }*//* EquateC */


static REALobject CAddC(REALobject instance, REALobject y)
{
	ComplexData		*xData, *yData, *zData;
	REALobject		z;
	
	if(instance==nil || y==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	xData = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	yData = (ComplexData *) REALGetClassData(y, &ComplexClass);
	
	if (xData == nil || yData == nil)
		return instance;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = xData->Re + yData->Re;
	zData->Im = xData->Im + yData->Im;
	
	return z;
	
}/* CAdd */


static REALobject CAddS(REALobject instance, double yRe)
{
	ComplexData		*xData, *zData;
	REALobject		z;
	
	if(instance==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	xData = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (xData == nil)
		return instance;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = xData->Re + yRe;
	zData->Im = xData->Im;
	
	return z;
	
}/* CAddS */



static REALobject CSubC(REALobject instance, REALobject y)
{
	ComplexData		*xData, *yData, *zData;
	REALobject		z;
	
	if(instance==nil || y==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	xData = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	yData = (ComplexData *) REALGetClassData(y, &ComplexClass);
	
	if (xData == nil || yData == nil)
		return instance;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = xData->Re - yData->Re;
	zData->Im = xData->Im - yData->Im;
	
	return z;
	
}/* CSub */


static REALobject CSubS(REALobject instance, double yRe)
{
	ComplexData		*xData, *zData;
	REALobject		z;
	
	if(instance==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	xData = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (xData == nil)
		return instance;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = xData->Re - yRe;
	zData->Im = xData->Im;
	
	return z;
	
}/* CSubS */


static REALobject SSubC(REALobject instance, double yRe)
{
	ComplexData		*xData, *zData;
	REALobject		z;
	
	if(instance==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	xData = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (xData == nil)
		return instance;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = yRe - xData->Re;
	zData->Im = -xData->Im;
	
	return z;
	
}/* SSubC */


static REALobject CMulC(REALobject instance, REALobject y)
{
	ComplexData		*xData, *yData, *zData;
	REALobject		z;
	
	if(instance==nil || y==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	xData = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	yData = (ComplexData *) REALGetClassData(y, &ComplexClass);
	
	if (xData == nil || yData == nil)
		return instance;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	complexMultiply(*zData, *xData, *yData);
	
	return z;
	
}/* CMulC */


static REALobject CMulS(REALobject instance, double yRe)
{
	ComplexData		*xData, *zData;
	REALobject		z;
	
	if(instance==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	xData = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (xData == nil)
		return instance;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = yRe*xData->Re;
	zData->Im = yRe*xData->Im;
	
	return z;
	
}/* CMulS */


static REALobject CDivC(REALobject instance, REALobject y)
{
	ComplexData		*xData, *yData, *zData;
	REALobject		z;
	
	if(instance==nil || y==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	xData = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	yData = (ComplexData *) REALGetClassData(y, &ComplexClass);
	
	if (xData == nil)
		return instance;
	
	if(yData == nil || (yData->Re==0 && yData->Im==0))
	{
		xData->Re = NAN;
		xData->Im = NAN;
		return instance;
	}
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	complexDivide(*zData, *xData, *yData);
	
	return z;
	
}/* CDivC */


static REALobject CDivS(REALobject instance, double yRe)
{
	ComplexData		*xData, *zData;
	REALobject		z;
	
	if(instance==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	xData = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (xData == nil)
		return instance;
	
	if(yRe==0)
	{
		xData->Re = NAN;
		xData->Im = NAN;
		return instance;
	}
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = xData->Re/yRe;
	zData->Im = xData->Im/yRe;
	
	return z;
	
}/* CDivS */


static REALobject SDivC(REALobject instance, double yRe)
{
	ComplexData		*xData, *zData;
	REALobject		z;
	double			den;
	
	if(instance==nil)
	{
		z = REALnewInstance("Complex");
		zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
		
		zData->Re = NAN;
		zData->Im = NAN;
		
		return z;
	}
	
	xData = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	
	if (xData == nil)
		return instance;
	
	if(xData->Re==0 && xData->Im==0)
	{
		xData->Re = NAN;
		xData->Im = NAN;
		return instance;
	}
	
	den = xData->Re*xData->Re + xData->Im*xData->Im;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = yRe*xData->Re/den;
	zData->Im = -yRe*xData->Im/den;
	
	return z;
	
}/* SDivC */


static int CompareCC(REALobject instance, REALobject x)
{
	ComplexData		*data, *xData;
	
	if(instance==nil || x==nil)
		return 0;
	
	data = (ComplexData *) REALGetClassData(instance, &ComplexClass);
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	
	if(data==nil || xData==nil)
		return 0;
	
	if(data->Re<xData->Re)
		return -1;
	
	if(data->Re>xData->Re)
		return 1;
	
	if(data->Im<xData->Im)
		return -1;
	
	if(data->Im>xData->Im)
		return 1;
	
	return 0;
	
}/* CompareCC */


// REALbasic class & method definitions for the Complex object

static REALmethodDefinition ComplexMethods[] = {
	// defines methods:
	{ (REALproc) ComplexReal, REALnoImplementation, "Real() as double", REALconsoleSafe },
	{ (REALproc) ComplexImag, REALnoImplementation, "Imag() as double", REALconsoleSafe },
	{ (REALproc) ComplexStr, REALnoImplementation, "Str() as String", REALconsoleSafe },
	{ (REALproc) ComplexStr2, REALnoImplementation, "Str2() as String", REALconsoleSafe },
	{ (REALproc) ComplexStr2EE, REALnoImplementation, "Str2EE() as String", REALconsoleSafe },
	{ (REALproc) ComplexConstructor, REALnoImplementation, "Constructor()", REALconsoleSafe },
	{ (REALproc) NewComplex, REALnoImplementation, "Constructor(xRe as double, xIm as double)", REALconsoleSafe },
	{ (REALproc) NewComplexD, REALnoImplementation, "Constructor(x as double)", REALconsoleSafe },
	{ (REALproc) NewComplexS, REALnoImplementation, "Constructor(inputString As String)", REALconsoleSafe },
	{ (REALproc) ComplexNegate, REALnoImplementation, "Operator_Negate() As Complex", REALconsoleSafe },
	{ (REALproc) CAddC, REALnoImplementation, "Operator_Add(y As Complex) As Complex", REALconsoleSafe },
	{ (REALproc) CAddS, REALnoImplementation, "Operator_Add(yRe As Double) As Complex", REALconsoleSafe },
	{ (REALproc) CAddS, REALnoImplementation, "Operator_AddRight(yRe As Double) As Complex", REALconsoleSafe },
	{ (REALproc) CSubC, REALnoImplementation, "Operator_Subtract(y As Complex) As Complex", REALconsoleSafe },
	{ (REALproc) CSubS, REALnoImplementation, "Operator_Subtract(yRe As Double) As Complex", REALconsoleSafe },
	{ (REALproc) SSubC, REALnoImplementation, "Operator_SubtractRight(yRe As Double) As Complex", REALconsoleSafe },
	{ (REALproc) CMulC, REALnoImplementation, "Operator_Multiply(y As Complex) As Complex", REALconsoleSafe },
	{ (REALproc) CMulS, REALnoImplementation, "Operator_Multiply(yRe As Double) As Complex", REALconsoleSafe },
	{ (REALproc) CMulS, REALnoImplementation, "Operator_MultiplyRight(yRe As Double) As Complex", REALconsoleSafe },
	{ (REALproc) CDivC, REALnoImplementation, "Operator_Divide(y As Complex) As Complex", REALconsoleSafe },
	{ (REALproc) CDivS, REALnoImplementation, "Operator_Divide(yRe As Double) As Complex", REALconsoleSafe },
	{ (REALproc) SDivC, REALnoImplementation, "Operator_DivideRight(yRe As Double) As Complex", REALconsoleSafe },
	{ (REALproc) CompareCC, REALnoImplementation, "Operator_Compare(x As Complex) As Integer", REALconsoleSafe }
};


REALclassDefinition ComplexClass = {
	kCurrentREALControlVersion,
	"Complex",                    			// name of class
	nil,                                    // no superclasses
	sizeof(ComplexData),                      // size of our data
	0,																			// --reserved--
	(REALproc) ComplexConstructor,     				// constructor
	(REALproc) ComplexDestructor,							// destructor
	nil,            												// properties
	0,																			// property count
	ComplexMethods,               					// methods
	sizeof(ComplexMethods) / sizeof(REALmethodDefinition),	// method count
	// other fields left nil -- no events etc.
};

// the following pragma is essential here

#pragma mark -


// other methods

static REALobject ComplexObjectNAN()
{
	REALobject		z;
	ComplexData		*zData;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = NAN;
	zData->Im = NAN;
	
	return z;
	
}/* ComplexObjectNAN */


static REALobject myI()
{
	REALobject		z;
	ComplexData		*zData;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = 0;
	zData->Im = 1;
	
	return z;
	
}/* myI */

/*
 static REALobject MyComplex(double xRe, double xIm)
 {
 REALobject		z;
 ComplexData		*zData;
 
 z = REALnewInstance("Complex");
 zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
 
 zData->Re = xRe;
 zData->Im = xIm;
 
 return z;
 
 }*//* MyComplex */

/*
 static REALobject NewComplexFromD(double x)
 {
 REALobject		z;
 ComplexData		*zData;
 
 z = REALnewInstance("Complex");
 zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
 
 zData->Re = x;
 zData->Im = 0;
 
 return z;
 
 }*//* NewComplexFromD */


static double AbsC(REALobject x)
{
	ComplexData		*xData;
	double			result;
	
	if(x==nil)
		return NAN;
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	
	if (xData == nil)
		return NAN;
	
	result = sqrt(xData->Re*xData->Re + xData->Im*xData->Im);
	
	return result;
	
}/* AbsC */


/*
 static double AbsD(double x)
 {
 if(x<0)
 x = -x;
 
 return x;
 
 }*//* AbsD */


/*
 static void ComplexAdd(REALobject z, REALobject x, REALobject y)
 {
 ComplexData		*xData, *yData, *zData;
 
 xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
 yData = (ComplexData *) REALGetClassData(y, &ComplexClass);
 zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
 
 if (xData == nil || yData == nil)
 return;
 
 if(zData == nil)
 {
 z = REALnewInstance("Complex");
 zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
 if (zData == nil)
 return;
 }
 
 zData->Re = xData->Re + yData->Re;
 zData->Im = xData->Im + yData->Im;
 
 }*//* ComplexAdd */


static REALobject ExpC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myExp(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* ExpC */


static REALobject LogC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	if(xData->Re==0 && xData->Im==0)
		return x;
	
	myLog(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* LogC */


static REALobject SinC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	mySin(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* SinC */


static REALobject CosC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myCos(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* CosC */


static REALobject TanC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myTan(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* TanC */


static REALobject SqrtC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	mySqrt(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* SqrtC */


static REALobject SqrC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	mySqr(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* SqrC */


// RB doesn't have Sqr of a double
static double SqrD(double x)
{
	return x*x;
	
}/* SqrD */


// RB doesn't have sinh of a double
static double SinhD(double x)
{
	double		result;
	
	mySinh(result, x);
	return result;
	
}/* SinhD */


static double CoshD(double x)
{
	double		result;
	
	myCosh(result, x);
	return result;
	
}/* CoshD */


static double TanhD(double x)
{
	double		theSinh, theCosh;
	
	mySinh(theSinh, x);
	theCosh = sqrt(1 + theSinh*theSinh);
	
	return theSinh/theCosh;
	
}/* TanhD */


static double aSinhD(double x)
{
	double		result;
	
	myASinh(result, x);
	
	return result;
	
}/* aSinhD */


static double aCoshD(double x)
{
	double		result;
	
	result = 0;
	myACosh(result, x);
	
	return result;
	
}/* aCoshD */


static double aTanhD(double x)
{
	double		result;
	
	result = 0;
	myATanh(result, x);
	
	return result;
	
}/* aTanhD */


static REALobject SinhC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	mySinhC(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* SinhC */


static REALobject CoshC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myCoshC(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* CoshC */


static REALobject TanhC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myTanhC(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* TanhC */


static REALobject aSinC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myASin(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* aSinC */


static REALobject aCosC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myACos(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* aCosC */


static REALobject aTanC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myATan(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* aTanC */


static REALobject aSinhC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myASinhC(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* aSinhC */


static REALobject aCoshC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myACoshC(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* aCoshC */


static REALobject aTanhC(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myATanhC(zRe, zIm, xData->Re, xData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* aTanhC */


static REALobject PowCC(REALobject x, REALobject y)
{
	ComplexData		*xData, *yData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil || y==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	yData = (ComplexData *) REALGetClassData(y, &ComplexClass);
	if (yData == nil)
		return y;
	
	myPowCC(zRe, zIm, xData->Re, xData->Im, yData->Re, yData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* PowCC */


static REALobject PowCD(REALobject x, double y)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	myPowCD(zRe, zIm, xData->Re, xData->Im, y);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* PowCD */


static REALobject PowDC(double x, REALobject y)
{
	ComplexData		*yData;
	REALobject		z;
	ComplexData		*zData;
	double			zRe, zIm;
	
	if(y==nil)
		return ComplexObjectNAN();
	
	yData = (ComplexData *) REALGetClassData(y, &ComplexClass);
	if (yData == nil)
		return y;
	
	myPowDC(zRe, zIm, x, yData->Re, yData->Im);
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = zRe;
	zData->Im = zIm;
	
	return z;
	
}/* PowDC */


static REALobject Conjugate(REALobject x)
{
	ComplexData		*xData;
	REALobject		z;
	ComplexData		*zData;
	
	if(x==nil)
		return ComplexObjectNAN();
	
	xData = (ComplexData *) REALGetClassData(x, &ComplexClass);
	if (xData == nil)
		return x;
	
	z = REALnewInstance("Complex");
	zData = (ComplexData *) REALGetClassData(z, &ComplexClass);
	
	zData->Re = xData->Re;
	zData->Im = -xData->Im;
	
	return z;
	
}/* Conjugate */


static void SetComplexPrecision(long precision)
{
	myPrecision = precision;
	if(myPrecision<3)
		myPrecision = 3;
	if(myPrecision>17)
		myPrecision = 17;
	
}/* SetComplexPrecision */


static REALmethodDefinition otherComplexMethods[] = {
	{ (REALproc) myI, REALnoImplementation, "_i() as Complex", REALconsoleSafe },
	{ (REALproc) myI, REALnoImplementation, "_j() as Complex", REALconsoleSafe },
	//{ (REALproc) MyComplex, REALnoImplementation, "Conv(xRe As Double, xIm As Double) As Complex", REALconsoleSafe },
	//{ (REALproc) NewComplexFromD, REALnoImplementation, "Complex(x As Double) As Complex", REALconsoleSafe },
	{ (REALproc) AbsC, REALnoImplementation, "Abs(x As Complex) As Double", REALconsoleSafe },
	//{ (REALproc) AbsD, REALnoImplementation, "Abs(x As Double) As Double", REALconsoleSafe },
	//{ (REALproc) ComplexAdd, REALnoImplementation, "Add(z as Complex, x as Complex, y as Complex)", REALconsoleSafe },
	{ (REALproc) ExpC, REALnoImplementation, "Exp(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) LogC, REALnoImplementation, "Log(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) SinC, REALnoImplementation, "Sin(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) CosC, REALnoImplementation, "Cos(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) TanC, REALnoImplementation, "Tan(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) SqrtC, REALnoImplementation, "Sqrt(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) SqrC, REALnoImplementation, "Sqr(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) SqrD, REALnoImplementation, "Sqr(x as Double) as Double", REALconsoleSafe },
	{ (REALproc) SinhD, REALnoImplementation, "Sinh(x as Double) as Double", REALconsoleSafe },
	{ (REALproc) CoshD, REALnoImplementation, "Cosh(x as Double) as Double", REALconsoleSafe },
	{ (REALproc) TanhD, REALnoImplementation, "Tanh(x as Double) as Double", REALconsoleSafe },
	{ (REALproc) aSinhD, REALnoImplementation, "aSinh(x as Double) as Double", REALconsoleSafe },
	{ (REALproc) aCoshD, REALnoImplementation, "aCosh(x as Double) as Double", REALconsoleSafe },
	{ (REALproc) aTanhD, REALnoImplementation, "aTanh(x as Double) as Double", REALconsoleSafe },
	{ (REALproc) SinhC, REALnoImplementation, "Sinh(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) CoshC, REALnoImplementation, "Cosh(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) TanhC, REALnoImplementation, "Tanh(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) aSinC, REALnoImplementation, "aSin(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) aCosC, REALnoImplementation, "aCos(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) aTanC, REALnoImplementation, "aTan(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) aSinhC, REALnoImplementation, "aSinh(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) aCoshC, REALnoImplementation, "aCosh(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) aTanhC, REALnoImplementation, "aTanh(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) PowCC, REALnoImplementation, "Pow(x as Complex, y as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) PowCD, REALnoImplementation, "Pow(x as Complex, y as Double) as Complex", REALconsoleSafe },
	{ (REALproc) PowDC, REALnoImplementation, "Pow(x as Double, y as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) Conjugate, REALnoImplementation, "Conjugate(x as Complex) as Complex", REALconsoleSafe },
	{ (REALproc) SetComplexPrecision, REALnoImplementation, "SetComplexPrecision(precision As Integer)", REALconsoleSafe }
};



void PluginEntry(void)
{
	int		i;
	
	REALRegisterClass(&ComplexClass);
	
	for (i = 0; i < sizeof(otherComplexMethods) / sizeof(REALmethodDefinition); ++i)
		REALRegisterMethod(&otherComplexMethods[i]);
	
	myPrecision = 7;
	
}
